/*
 * \file   TI_MSPBoot_AppMgr.c
 *
 * \brief  Application Manager. Handles App validation, decides if device
 *         should jump to App or stay in bootloader
 *         This file supports dual image
 */
/* --COPYRIGHT--,BSD
 * Copyright (c) 2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/

//
// Include files
//
#include <stdbool.h>
#include "msp430.h"
#include "TI_MSPBoot_Common.h"
#include "TI_MSPBoot_AppMgr.h"
#include "crc.h"

//
//  Configuration checks
//
#if ((CONFIG_APPMGR_APP_VALIDATE != 2) && (CONFIG_APPMGR_APP_VALIDATE != 3))
#error CRC is required in order to validate application
#endif

//
//  Global variables
//
/*! Password sent by Application to force boot mode. This variable is in a fixed
    location and should keep same functionality and location in Boot and App */
extern uint16_t  PassWd;

/*! Status and Control byte. This variable is in a fixed
 location and should keep same functionality and location in Boot and App */
extern uint8_t  StatCtrl;

//
//  Local function prototypes
//
static bool TI_MSPBoot_AppMgr_BootisForced(void);
static void TI_MSPBoot_Restore_DefaultClockSettings(void);

/******************************************************************************
 *
 * @brief   Checks if an Application is valid
 *  Depending on CONFIG_APPMGR_APP_VALIDATE, this function can validate app by:
 *  CONFIG_APPMGR_APP_VALIDATE  |    Function
 *          2                   | Check CRC_CCITT of application and compare vs
 *                              |  an expected value
 *          other               | Application is expected to be valid
 *
* @param Downarea 0:Valdate App Area, 1:Validate Download Area
 *
 * @return  true if application is valid,
 *          false if applicaiton is invalid
 *****************************************************************************/
#if (CONFIG_APPMGR_APP_VALIDATE == 2)
// Check the Applications's Checksum
static bool TI_MSPBoot_AppMgr_AppisValid(uint8_t Downarea)
{
    extern uint16_t _Appl_Reset_Vector;
    extern uint16_t _Down_Checksum;   // Address of Download area checksum
    extern uint16_t _Down_Start_Memory;     // Address of Download area start
    extern uint16_t _Down_CRC_Size1;

    static uint16_t result = 0;
    
    if (Downarea == 0)
    {
        // Check if Application Reset vector exists
        if (*(volatile uint16_t *)(&_Appl_Reset_Vector) != 0xFFFF)
        {
            return true;
        }
        else
        {
            return false;
        }

    }
    else
    {
        // Validate Download Area
        // calculate CRC and compare vs expected value in reserved location
        result = crc16MakeBitwise((uint8_t*) &_Down_Start_Memory, (uint16_t) &_Down_CRC_Size1);

        if (result != _Down_Checksum) return false;
        else return true;
    }
}
#else
// Always assume that Application is valid
#error "Application is not validated"
#endif

/******************************************************************************
 *
 * @brief   Decides whether to stay in MSPBoot or if it should jump to App
 *  MSPBoot:  Boot mode is forced by a call from App, OR
 *          Boot mode is forced by an external event (button pressed), OR
 *          Application is invalid
 *  App:    Boot mode is not forced, AND
 *          Application is valid
 *
 * @return  true if application is valid and should be executed
 *          false if we must stay in Boot mode
 *****************************************************************************/
bool TI_MSPBoot_AppMgr_ValidateApp(void)
{
    extern void TI_MSPBoot_MI_EraseAppDirect(uint8_t DownArea);
    extern void TI_MSPBoot_MI_ReplaceApp(void);

    if (TI_MSPBoot_AppMgr_BootisForced() == false)
    {
        // If Bootloader is not forced, check the current image
        if (TI_MSPBoot_AppMgr_AppisValid(0) == true)
        {
            // Image valid, jump to App
            TI_MSPBoot_Restore_DefaultClockSettings();
            return true;
        }
        else
        {
            // Image invalid, check Download Area
            if (TI_MSPBoot_AppMgr_AppisValid(1) == true)
            {
                // Erase App Area
                TI_MSPBoot_MI_EraseAppDirect(0);
                // Replace the Image in App Area by Download Area
                TI_MSPBoot_MI_ReplaceApp();
                // Validate image in App Area
                if (TI_MSPBoot_AppMgr_AppisValid(0) == true)
                {
                    // Erase the image in Download Area
                    TI_MSPBoot_MI_EraseAppDirect(1);
                    TI_MSPBoot_Restore_DefaultClockSettings();
                    return true;
                }
            }
        }
    }
    
    return false; // Boot is forced or App is valid
}


/******************************************************************************
 *
 * @brief   Jump to Appplication 
 *          The image in download area is validated and copied to App Area if valid
 *          A Reset is forced in order to validate Application after Reset
 *
 * @return  None
 *****************************************************************************/
void TI_MSPBoot_AppMgr_JumpToApp(void)
{
    extern void TI_MSPBoot_MI_EraseAppDirect(uint8_t DownArea);
    extern void TI_MSPBoot_MI_ReplaceApp(void);

    // Check image in download area 
#ifdef NDEBUG
    // Feed the dog
    WATCHDOG_FEED();
#endif
    if (TI_MSPBoot_AppMgr_AppisValid(1) == true)
    {
        // If Image is valid, erase current App
        TI_MSPBoot_MI_EraseAppDirect(0);
        // Replace current App by Image in Download Area
        TI_MSPBoot_MI_ReplaceApp();
#ifdef NDEBUG
        // Feed the dog 
        WATCHDOG_FEED();
#endif

        // Validate image in App Area
        if (TI_MSPBoot_AppMgr_AppisValid(0) == true)
        {
            // Erase the image in Download Area
             TI_MSPBoot_MI_EraseAppDirect(1);
        }
    }
#if defined (HW_RESET_BOR)
    // Force a Software BOR
    PMMCTL0 = PMMPW | PMMSWBOR;
#else
    // Force a PUC reset by writing incorrect Watchdog password
    WDTCTL = 0xDEAD;
#endif
}


/******************************************************************************
 *
 * @brief   Checks if Boot mode is forced
 *          Boot mode is forced by an application call sending a request and password
 *          Or by an external event such as a button press
 *
 * @return  true Boot mode is forced
 *          false Boot mode is not forced
 *****************************************************************************/
static bool TI_MSPBoot_AppMgr_BootisForced(void)
{
    bool ret = false;

    // Check if application is requesting Boot mode and password is correct
    if (((StatCtrl & BOOT_APP_REQ) != 0) && (PassWd == BSL_PASSWORD))
    {
        ret = true;
    }

    // Check for an external event such as S1 (P1.3) button in Launchpad
    __delay_cycles(10000);   // Wait for pull-up to go high
    //If S1 button (P1.3) is pressed, force BSL
    if (HW_ENTRY_CONDITION)
    {
        ret = true;
    }

    // Clear Status and Control byte and password
    PassWd = 0;
    StatCtrl = 0;
    return ret;
}

static void TI_MSPBoot_Restore_DefaultClockSettings(void)
{
    // Set all clock registers to default value
    // Validate the calibration values (only in Debug)
    ASSERT_H((CALBC1_1MHZ !=0xFF) && (CALDCO_1MHZ != 0xFF));

    BCSCTL1 = CALBC1_1MHZ;              // Set DCO to default of 1Mhz
    DCOCTL = CALDCO_1MHZ;
}
